<?php

/**
 * StatusPage Classic Widget.
 *
 * Legacy widget for themes that use widget areas.
 *
 * @package StatusPageWidget
 */

// Exit if accessed directly.
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Class StatusPage_Classic_Widget
 *
 * Classic WordPress widget for displaying status widgets.
 */
class StatusPage_Classic_Widget extends WP_Widget
{

    /**
     * Constructor.
     */
    public function __construct()
    {
        parent::__construct(
            'statuspage_widget',
            __('StatusPage Widget', 'status-page-widget'),
            array(
                'description'                 => __('Display a status page widget from StatusPage.me.', 'status-page-widget'),
                'customize_selective_refresh' => true,
                'show_instance_in_rest'       => true,
            )
        );
    }

    /**
     * Front-end display of widget.
     *
     * @param array $args     Widget arguments.
     * @param array $instance Saved values from database.
     */
    public function widget($args, $instance)
    {
        echo $args['before_widget'];

        if (! empty($instance['title'])) {
            echo $args['before_title'] . apply_filters('widget_title', $instance['title']) . $args['after_title'];
        }

        $render_atts = array(
            'slug'           => ! empty($instance['slug']) ? $instance['slug'] : '',
            'theme'          => ! empty($instance['theme']) ? $instance['theme'] : 'auto',
            'compact'        => ! empty($instance['compact']),
            'show_incidents' => ! empty($instance['show_incidents']),
            'rtm'            => isset($instance['rtm']) ? absint($instance['rtm']) : 60,
            'upd'            => isset($instance['upd']) ? absint($instance['upd']) : 7,
            'disable_link'   => ! empty($instance['disable_link']),
            'render_mode'    => ! empty($instance['render_mode']) ? $instance['render_mode'] : 'js',
        );

        echo StatusPage_Renderer::render($render_atts);

        echo $args['after_widget'];
    }

    /**
     * Back-end widget form.
     *
     * @param array $instance Previously saved values from database.
     */
    public function form($instance)
    {
        $title          = ! empty($instance['title']) ? $instance['title'] : '';
        $slug           = ! empty($instance['slug']) ? $instance['slug'] : '';
        $theme          = ! empty($instance['theme']) ? $instance['theme'] : 'auto';
        $compact        = ! empty($instance['compact']);
        $show_incidents = ! empty($instance['show_incidents']);
        $rtm            = isset($instance['rtm']) ? absint($instance['rtm']) : 60;
        $upd            = isset($instance['upd']) ? absint($instance['upd']) : 7;
        $disable_link   = ! empty($instance['disable_link']);
        $render_mode    = ! empty($instance['render_mode']) ? $instance['render_mode'] : 'js';
?>
        <p>
            <label for="<?php echo esc_attr($this->get_field_id('title')); ?>">
                <?php esc_html_e('Title:', 'status-page-widget'); ?>
            </label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('title')); ?>"
                name="<?php echo esc_attr($this->get_field_name('title')); ?>" type="text"
                value="<?php echo esc_attr($title); ?>">
        </p>

        <p>
            <label for="<?php echo esc_attr($this->get_field_id('slug')); ?>">
                <?php esc_html_e('Status Page Slug:', 'status-page-widget'); ?>
            </label>
            <input class="widefat" id="<?php echo esc_attr($this->get_field_id('slug')); ?>"
                name="<?php echo esc_attr($this->get_field_name('slug')); ?>" type="text"
                value="<?php echo esc_attr($slug); ?>" required>
            <small><?php esc_html_e('Your StatusPage.me slug', 'status-page-widget'); ?></small>
        </p>

        <p>
            <label for="<?php echo esc_attr($this->get_field_id('theme')); ?>">
                <?php esc_html_e('Theme:', 'status-page-widget'); ?>
            </label>
            <select class="widefat" id="<?php echo esc_attr($this->get_field_id('theme')); ?>"
                name="<?php echo esc_attr($this->get_field_name('theme')); ?>">
                <option value="auto" <?php selected($theme, 'auto'); ?>><?php esc_html_e('Auto (System)', 'status-page-widget'); ?></option>
                <option value="light" <?php selected($theme, 'light'); ?>><?php esc_html_e('Light', 'status-page-widget'); ?></option>
                <option value="dark" <?php selected($theme, 'dark'); ?>><?php esc_html_e('Dark', 'status-page-widget'); ?></option>
            </select>
        </p>

        <p>
            <label for="<?php echo esc_attr($this->get_field_id('render_mode')); ?>">
                <?php esc_html_e('Render Mode:', 'status-page-widget'); ?>
            </label>
            <select class="widefat" id="<?php echo esc_attr($this->get_field_id('render_mode')); ?>"
                name="<?php echo esc_attr($this->get_field_name('render_mode')); ?>">
                <option value="js" <?php selected($render_mode, 'js'); ?>><?php esc_html_e('JavaScript (recommended)', 'status-page-widget'); ?></option>
                <option value="php" <?php selected($render_mode, 'php'); ?>><?php esc_html_e('PHP (server-side)', 'status-page-widget'); ?></option>
            </select>
            <small><?php esc_html_e('JS mode supports auto-refresh; PHP mode is better for SEO.', 'status-page-widget'); ?></small>
        </p>

        <p>
            <input class="checkbox" type="checkbox" id="<?php echo esc_attr($this->get_field_id('compact')); ?>"
                name="<?php echo esc_attr($this->get_field_name('compact')); ?>" <?php checked($compact); ?>>
            <label for="<?php echo esc_attr($this->get_field_id('compact')); ?>">
                <?php esc_html_e('Compact Mode', 'status-page-widget'); ?>
            </label>
        </p>

        <p>
            <input class="checkbox" type="checkbox" id="<?php echo esc_attr($this->get_field_id('show_incidents')); ?>"
                name="<?php echo esc_attr($this->get_field_name('show_incidents')); ?>" <?php checked($show_incidents); ?>>
            <label for="<?php echo esc_attr($this->get_field_id('show_incidents')); ?>">
                <?php esc_html_e('Show Active Incidents', 'status-page-widget'); ?>
            </label>
        </p>

        <p>
            <input class="checkbox" type="checkbox" id="<?php echo esc_attr($this->get_field_id('disable_link')); ?>"
                name="<?php echo esc_attr($this->get_field_name('disable_link')); ?>" <?php checked($disable_link); ?>>
            <label for="<?php echo esc_attr($this->get_field_id('disable_link')); ?>">
                <?php esc_html_e('Disable Link to Status Page', 'status-page-widget'); ?>
            </label>
        </p>

        <p>
            <label for="<?php echo esc_attr($this->get_field_id('rtm')); ?>">
                <?php esc_html_e('Response Time (minutes):', 'status-page-widget'); ?>
            </label>
            <input class="tiny-text" id="<?php echo esc_attr($this->get_field_id('rtm')); ?>"
                name="<?php echo esc_attr($this->get_field_name('rtm')); ?>" type="number"
                min="0" max="240" step="1" value="<?php echo esc_attr($rtm); ?>">
            <small><?php esc_html_e('0-240 (0 = hide)', 'status-page-widget'); ?></small>
        </p>

        <p>
            <label for="<?php echo esc_attr($this->get_field_id('upd')); ?>">
                <?php esc_html_e('Uptime Days:', 'status-page-widget'); ?>
            </label>
            <input class="tiny-text" id="<?php echo esc_attr($this->get_field_id('upd')); ?>"
                name="<?php echo esc_attr($this->get_field_name('upd')); ?>" type="number"
                min="0" max="30" step="1" value="<?php echo esc_attr($upd); ?>">
            <small><?php esc_html_e('0-30 (0 = hide)', 'status-page-widget'); ?></small>
        </p>
<?php
    }

    /**
     * Sanitize widget form values as they are saved.
     *
     * @param array $new_instance Values just sent to be saved.
     * @param array $old_instance Previously saved values from database.
     * @return array Updated safe values to be saved.
     */
    public function update($new_instance, $old_instance)
    {
        $instance = array();

        $instance['title']          = ! empty($new_instance['title']) ? sanitize_text_field($new_instance['title']) : '';
        $instance['slug']           = ! empty($new_instance['slug']) ? sanitize_text_field($new_instance['slug']) : '';
        $instance['theme']          = in_array($new_instance['theme'], array('auto', 'light', 'dark'), true) ? $new_instance['theme'] : 'auto';
        $instance['render_mode']    = in_array($new_instance['render_mode'], array('js', 'php'), true) ? $new_instance['render_mode'] : 'js';
        $instance['compact']        = ! empty($new_instance['compact']);
        $instance['show_incidents'] = ! empty($new_instance['show_incidents']);
        $instance['disable_link']   = ! empty($new_instance['disable_link']);
        $instance['rtm']            = min(absint($new_instance['rtm']), 240);
        $instance['upd']            = min(absint($new_instance['upd']), 30);

        return $instance;
    }
}
